#!/usr/bin/env wish8.5
#
# StuDNA View, by Martin Saturka
# studna wellspring results survey
# available under the MIT/X license
# http://www.bioplexity.org/
# version 2.0.10 (2009 / August)
#

set stversion "2.0.10"

catch {
  package require tile
  ::ttk::setTheme clam
}

set topw .
set topname "StuDNA view"
wm title $topw $topname
wm geometry $topw "840x560"
wm minsize $topw 640 480
wm resizable $topw yes yes
wm protocol $topw WM_DELETE_WINDOW {wantToExit}

set loaddir ""
if {0 < $argc} {
  foreach argdata $argv {
    if {1 == [file isdirectory $argdata]} {
      set loaddir $argdata
    }
    if {1 == [file isfile $argdata]} {
      set loaddir [file dirname $argdata]
    }
  }
}

#-----------------------------------------

array set sortedesd {}
array set sorteddse {}
array set sortedsde {}
array set sortedsed {}

set inccaselen 0
set inccases [list]
set inctuplets [list]
set incgapslen 0
set incgaplist [list]
set incrowlen 0
set incrowvis 17
set incinilist [list]
set inccurlist [list]

set incusegap -1

set totLines 0

set curestims [list]

set probsCounts [list]
set probsSingle [list]
set probsBonfer [list]

#-----------------------------------------

set fontFixed [font create]
font configure $fontFixed -size 10
font configure $fontFixed -family "Courier"

set fontValues [font create]
font configure $fontValues -size 9

set fontLesser [font create]
font configure $fontLesser -size 8

set fontLarger [font create]
font configure $fontLarger -size 16

#-----------------------------------------

proc ::tk_textCopy {{wdg ""} {rectangle 0} {seqonly 0}} {

  if {"" == $wdg} {
    return
  }

  set selranges [list]
  set bad [catch {set selranges [$wdg tag ranges sel]}]
  if {$bad} {
    return
  }

  set firstLine [lindex $selranges 0]
  set firstIndex [lindex [split $firstLine "."] 1]
  set lastLine [lindex $selranges end]
  set lastIndex [lindex [split $lastLine "."] 1]
  incr lastIndex -1
  set reversed 0
  if {1 == $rectangle} {
    if {$lastIndex < $firstIndex} {
      set auxIndexList [list $lastIndex $firstIndex]
      set firstIndex [lindex $auxIndexList 0]
      set lastIndex [lindex $auxIndexList 1]
      incr firstIndex
      incr lastIndex -1
      set reversed 1
    }
  }

  set newline "\n"
  if {0 == [string compare -nocase -length 3 "win" [tk windowingsystem]]} {
    set newline "\r\n"
  }
  if {0 == [string compare -nocase -length 5 "classic" [tk windowingsystem]]} {
    set newline "\r"
  }
  set addnl 0

  set seldata ""
  set isFirst 1
  set lengthRanges [expr {[llength $selranges] / 2}]
  set currRangeInd 0
  foreach {start stop} $selranges {
    incr currRangeInd
    set startline [lindex [split $start "."] 0]
    set stopline [lindex [split $stop "."] 0]
    if {$startline != $stopline} {
      set addnl 1
    }
    set selarea [string trimright [$wdg get $start $stop]]
    set sellines [split $selarea "\n"]
    set curlines [list]
    set lengthLines [llength $sellines]
    set currLineInd 0
    foreach oneline $sellines {
      incr currLineInd
      set genedata 0
      if {1 == $rectangle} {
        if {0 == $reversed} {
          if {1 == $isFirst} {
            if {1 < $lengthLines} {
              set tryLineStart [join [list $startline ".0"] ""]
              set tryLineEnd [join [list $startline ".end"] ""]
              set tryLine [string trimright [$wdg get $tryLineStart $tryLineEnd]]
              set isGeneData 0
              if [string is alpha -strict $tryLine] {
                if [string is ascii -strict $tryLine] {
                  set isGeneData 1
                }
              }
              if {0 == $isGeneData} {
                set oneline $tryLine
              }
            }
          }
        }
      }
      set oneline [string trimright $oneline]
      if {1 == $rectangle} {
        set useFirstIndex $firstIndex
        set useLastIndex $lastIndex
        if {1 == $isFirst} {
          set useFirstIndex 0
          set useLastIndex [expr {$lastIndex - $firstIndex}]
        }
        if [string is alpha -strict $oneline] {
          if [string is ascii -strict $oneline] {
            set oneline [string range $oneline $useFirstIndex $useLastIndex]
            set genedata 1
          }
        }
      }
      set toAdd 1
      if {1 == $reversed} {
        if {1 == $isFirst} {
          set toAdd 0
        }
        if {$currRangeInd == $lengthRanges} {
          if {$currLineInd == $lengthLines} {
            set toAdd 0
          }
        }
      }
      if {1 == $seqonly} {
        if {0 == $genedata} {
          set toAdd 0
        }
      }
      if {1 == $toAdd} {
        lappend curlines $oneline
      }
      set isFirst 0
    }
    if {"" != $seldata} {
      append seldata $newline
      set addnl 1
    }
    append seldata [join $curlines $newline]
    if {1 < [llength $curlines]} {
      set addnl 1
    }
  }

  if {1 == $addnl} {
    append seldata $newline
  }

  clipboard clear -displayof $wdg
  clipboard append -displayof $wdg $seldata

}

proc copyEstimates {{wdg ""}} {
  global probsCounts
  global curestims
  global adjests
  global prDefFile

  if {"" == $wdg} {
    return
  }

  clipboard clear -displayof $wdg
  set rowcount [llength $probsCounts]
  if {0 == $rowcount} {
    return
  }

  set newline "\n"
  if {0 == [string compare -nocase -length 3 "win" [tk windowingsystem]]} {
    set newline "\r\n"
  }
  if {0 == [string compare -nocase -length 5 "classic" [tk windowingsystem]]} {
    set newline "\r"
  }

  set seldata "#studna estimations: "
  if {0 == $adjests} {
    append seldata "without adjustment"
  } else {
    append seldata "with adjustment"
  }
  append seldata $newline
  append seldata "#" $prDefFile
  append seldata $newline
  append seldata "count,exact,single,double"
  append seldata $newline

  for {set rind 0} {$rind < $rowcount} {incr rind} {
    set curline ""
    set curind $rind
    set curcnt [lindex $probsCounts $curind]
    append curline $curcnt

    set currow [lindex $curestims $curind]
    set est0 [lindex $currow 0]
    if [string is double -strict $est0] {
      set est0 [format "%1.6e" $est0]
    }
    set est1 [lindex $currow 1]
    if [string is double -strict $est1] {
      set est1 [format "%1.6e" $est1]
    }
    set est2 [lindex $currow 2]
    if [string is double -strict $est2] {
      set est2 [format "%1.6e" $est2]
    }
    append curline "," $est0 "," $est1 "," $est2

    append curline $newline
    append seldata $curline
  }

  clipboard append -displayof $wdg $seldata
}

proc copyIncidences {{wdg ""}} {
  global incusegap
  global gaplabel
  global sorttype
  global frDefFile

  global inccases
  global inctuplets
  global inccurlist

  if {"" == $wdg} {
    return
  }

  clipboard clear -displayof $wdg
  if {-1 == $incusegap} {
    return
  }

  set newline "\n"
  if {0 == [string compare -nocase -length 3 "win" [tk windowingsystem]]} {
    set newline "\r\n"
  }
  if {0 == [string compare -nocase -length 5 "classic" [tk windowingsystem]]} {
    set newline "\r"
  }

  set curgap [$gaplabel cget -text]
  set seldata "#studna incidences: gap $curgap, sorting: "
  if {0 == $sorttype} {
    append seldata "loaded"
  }
  if {1 == $sorttype} {
    append seldata "e-s-d"
  }
  if {2 == $sorttype} {
    append seldata "d-s-e"
  }
  if {3 == $sorttype} {
    append seldata "s-e-d"
  }
  if {4 == $sorttype} {
    append seldata "s-d-e"
  }
  append seldata $newline
  append seldata "#" $frDefFile
  append seldata $newline
  append seldata "tuplet,exact,single,double"
  append seldata $newline

  set listLength [llength $inccurlist]
  for {set rind 0} {$rind < $listLength} {incr rind} {
    set curline ""
    set curind [lindex $inccurlist $rind]
    set curtpl [lindex $inctuplets $curind]
    append curline $curtpl

    set curcase [lindex $inccases $curind]
    foreach cind [list 1 2 3] {
      set curcol [expr {(3 * $incusegap) + $cind - 1}]
      set curval [lindex $curcase $curcol]

      append curline "," $curval
    }

    append curline $newline
    append seldata $curline
  }

  clipboard append -displayof $wdg $seldata
}

#-----------------------------------------

set fmenu [frame .m -relief groove -bd 2]
pack $fmenu -side top -expand no -fill x

button $fmenu.mfile -text "Data files" -width 10 -command {DrawFile} -bd 1
button $fmenu.mestm -text "Estimates" -width 10 -command {DrawEstm} -bd 1
button $fmenu.mincd -text "Incidences" -width 10 -command {DrawIncd} -bd 1
button $fmenu.mseqn -text "Sequences" -width 10 -command {DrawSeqn} -bd 1
button $fmenu.mhelp -text "StuDNA" -width 10 -command {DrawAbout} -bd 1
set btlist [list "mfile" "mestm" "mincd" "mseqn" "mhelp"]

pack $fmenu.mfile $fmenu.mestm $fmenu.mincd $fmenu.mseqn -side left
pack $fmenu.mhelp -side right

set topf [frame .f -pady 2]
pack $topf -side top -expand yes -fill both

set pfcurrent "none"

#-----------------------------------------


set probvis 17
set sw [frame $topf.sw]
proc prepareEstm {sw} {
  global proboff
  global probscale

  global fontValues
  global fontLesser
  global probvis

  global adjests
  global swadesc

  set swa [frame $sw.a]
  set swi [frame $sw.i]
  set swb [frame $sw.b]
  pack $swa $swi $swb -side left

  set rdesc1 "  per row values:"
  set rdesc2 "Probability estimates\nto get count-or-more\ncases by chance"
  set rdesc3 " "
  set rdesd1 "\n"
  set rdesd2 "Adjusted estimation\nvalues for multiple\ngap search are via\nstrict adjustments.\n- Bonferroni method"
  set rdesd3 "\n"
  label $swa.lr1 -text $rdesc1 -justify left -anchor "w"
  label $swa.lr2 -text $rdesc2 -justify left
  label $swa.lr3 -text $rdesc3 -height 2 -justify left

  set adjests 0
  checkbutton $swa.cb -text "with adjustment" -variable adjests -command selectEstims -bd 1
  set swadesc [label $swa.desc -text "" -width 18 -height 3]
  label $swa.i0 -text " " -height 1

  label $swa.ls1 -text $rdesd1 -justify left -anchor "w"
  label $swa.ls2 -text $rdesd2 -justify left -font $fontLesser
  label $swa.ls3 -text $rdesd3 -justify left

  label $swa.i1 -text " " -height 2

  pack $swa.lr1 -side top -expand "no" -fill "x"
  pack $swa.lr2 $swa.lr3 -side top

  pack $swa.cb -side top
  pack $swa.desc $swa.i0 -side top -fill "x"

  pack $swa.ls1 -side top -expand "no" -fill "x"
  pack $swa.ls2 $swa.ls3 $swa.i1 -side top

  label $swi.l1 -text " " -width 6
  pack $swi.l1 -side left

  set swbm [frame $swb.matrix]
  set swbs [frame $swb.fscale]
  pack $swbm $swbs -side left

  label $swbm.ht -text "counts\n "
  label $swbm.h0 -text "exact\nmatch" -justify "left"
  label $swbm.h1 -text "w/ single\nmismatch" -justify "left"
  label $swbm.h2 -text "w/ double\nmismatch" -justify "left"
  grid $swbm.ht -column 0 -row 0
  grid $swbm.h0 -column 1 -row 0
  grid $swbm.h1 -column 2 -row 0
  grid $swbm.h2 -column 3 -row 0

  set rowcount $probvis
  set rowlist [list]
  for {set ind 1} {$ind <= $rowcount} {incr ind} {
    lappend rowlist $ind
  }

  foreach rind $rowlist {
    global cnt$rind
    set cnt$rind ""
    set cell [entry $swbm.d-r$rind -textvariable cnt$rind -width 6 -bd 1]
    $cell configure -state "readonly" -readonlybackground "#ffffff" -disabledforeground "#000000" -font $fontValues
    grid $cell -column 0 -row $rind
    bind $cell <Control-a> [list copyEstimates $cell]
    bind $cell <Control-A> [list copyEstimates $cell]
  }
  foreach cind [list 1 2 3] {
    foreach rind $rowlist {
      global est$rind-$cind
      set est$rind-$cind ""
      set cell [entry $swbm.d-r$rind-c$cind -textvariable est$rind-$cind -width 11 -bd 1]
      $cell configure -state "readonly" -readonlybackground "#ffffff" -disabledforeground "#000000" -font $fontValues
      grid $cell -column $cind -row $rind
      bind $cell <Control-a> [list copyEstimates $cell]
      bind $cell <Control-A> [list copyEstimates $cell]
    }
  }

  set proboff 0
  label $swbs.i0 -text " " -width 2
  set probscale [scale $swbs.rscale -orient "vertical" -length 200 -from 0 -to 0 -variable proboff -command setProbOffset]
  pack $swbs.i0 $swbs.rscale -side left
  $probscale configure -bigincrement 10

  bind $swbs.rscale <1> [list focus $swbs.rscale]
  bind $swbs.rscale <Shift-KeyPress> "scrollEsts 1 %K"
  bind $swbs.rscale <KeyPress> "scrollEsts 0 %K"

}
prepareEstm $sw

proc scrollEsts {shift what} {
  global probscale
  global proboff

  set toscroll 0
  if {0 == [string compare -nocase "Up" $what]} {
    if {1 == $shift} {
      set toscroll 1
    }
  }
  if {0 == [string compare -nocase "Left" $what]} {
    if {1 == $shift} {
      set toscroll 1
    }
  }
  if {0 == [string compare -nocase "Down" $what]} {
    if {1 == $shift} {
      set toscroll 2
    }
  }
  if {0 == [string compare -nocase "Right" $what]} {
    if {1 == $shift} {
      set toscroll 2
    }
  }
  if {0 == [string compare -nocase "Prior" $what]} {
    set toscroll 3
  }
  if {0 == [string compare -nocase "Next" $what]} {
    set toscroll 4
  }

  if {0 == $toscroll} {
    return
  }

  set sfrom [lindex [split [$probscale cget "-from"] "."] 0]
  set sto [lindex [split [$probscale cget "-to"] "."] 0]

  set aux 0

  if {1 == $toscroll} {
    set aux [expr {$proboff - 9}]
    if {$sfrom > $aux} {
      set aux $sfrom
    }
    set proboff $aux
  }
  if {3 == $toscroll} {
    set aux [expr {$proboff - 100}]
    if {$sfrom > $aux} {
      set aux $sfrom
    }
    set proboff $aux
  }

  if {2 == $toscroll} {
    set aux [expr {$proboff + 9}]
    if {$sto < $aux} {
      set aux $sto
    }
    set proboff $aux
  }
  if {4 == $toscroll} {
    set aux [expr {$proboff + 100}]
    if {$sto < $aux} {
      set aux $sto
    }
    set proboff $aux
  }

  setProbOffset $aux

}


proc DrawEstm {} {
  global topf
  global pfcurrent
  global sw
  global isAsking
  global fmenu
  global btlist

  global prDefFile
  global topname
  global topw

  if {1 == $isAsking} {
    return
  }

  set nametoset [file tail $prDefFile]
  set strset ""
  if {0 < [string length $nametoset]} {
    set strset "$nametoset - "
  }
  append strset $topname
  wm title $topw $strset

  foreach butt $btlist {
    $fmenu.$butt configure -relief "raised"
  }
  $fmenu.mestm configure -relief "sunken"

  if {"none" != $pfcurrent} {
    pack forget $pfcurrent
  }
  set pfcurrent $sw
  pack $sw -expand yes -fill none -side top

}

proc setProbOffset {newValue} {
  global proboff
  global probvis

  global probsCounts
  global curestims

  set rowcount [llength $probsCounts]

  set userows $probvis
  if {$userows > $rowcount} {
    set userows $rowcount
  }

  for {set rind 1} {$rind <= $probvis} {incr rind} {
    global cnt$rind
    global est$rind-1
    global est$rind-2
    global est$rind-3
  }

  if {0 == $rowcount} {
    for {set rind 1} {$rind <= $probvis} {incr rind} {
      set cnt$rind ""
      set est$rind-1 ""
      set est$rind-2 ""
      set est$rind-3 ""
    }
    return
  }

  for {set rind 1} {$rind <= $userows} {incr rind} {
    set curind [expr {$rind + $proboff - 1}]
    set cnt$rind [lindex $probsCounts $curind]
    set currow [lindex $curestims $curind]

    set est0 [lindex $currow 0]
    if [string is double -strict $est0] {
      set est0 [format "%1.4e" $est0]
    }
    set est1 [lindex $currow 1]
    if [string is double -strict $est1] {
      set est1 [format "%1.4e" $est1]
    }
    set est2 [lindex $currow 2]
    if [string is double -strict $est2] {
      set est2 [format "%1.4e" $est2]
    }

    set est$rind-1 $est0
    set est$rind-2 $est1
    set est$rind-3 $est2
  }

}

proc selectEstims {} {
  global probsCounts
  global probsSingle
  global probsBonfer

  global curestims
  global adjests
  global swadesc

  if {0 == $adjests} {
    set curestims $probsSingle
  } else {
    set curestims $probsBonfer
  }

  set curdesc "for a single gap"
  if {1 == $adjests} {
    set curdesc "adjusted values"
  }
  $swadesc configure -text "\ncurrent set:\n$curdesc"

  setProbOffset -1

}
selectEstims

proc updateEstims {} {
  global probsCounts
  global probsSingle
  global probsBonfer

  global curestims
  global adjests
  global proboff
  global probscale
  global probvis

  if {0 == $adjests} {
    set curestims $probsSingle
  } else {
    set curestims $probsBonfer
  }

  set proboff 0

  set rowcount [llength $probsCounts]
  set toval [expr {$rowcount - $probvis}]
  if {0 > $toval} {
    set toval 0
  }
  $probscale configure "-to" $toval

  setProbOffset 0

}

#-----------------------------------------


set lw [frame $topf.lw]
proc prepareIncd {lw} {
  global tuploff
  global tuploffscale
  global sorttype
  global selgap
  global selgapbox
  global gaplabel
  global incrowvis
  global fontFixed
  global fontValues

  set lwa [frame $lw.a]
  set lwi [frame $lw.i]
  set lwb [frame $lw.b]
  pack $lwa $lwi $lwb -side left

  set rdesc1 "gap lengths "
  label $lwa.lr1 -text $rdesc1 -justify left -anchor "w"
  set lwaf [frame $lwa.gf]
  set lwac [frame $lwa.cr]
  label $lwa.i1 -text " " -height 2
  set lwas [frame $lwa.st]

  pack $lwa.lr1 $lwaf $lwac $lwa.i1 $lwas -side top

  set selgap [list]
  set selgapbox [listbox $lwaf.lb -height 10 -width 10 -listvariable selgap -yscrollcommand "$lwaf.sb set" -selectmode browse -bd 1]
  $selgapbox configure -font $fontValues
  scrollbar $lwaf.sb -command "$lwaf.lb yview" -bd 1
  $selgapbox configure -background "#ffffff"

  bind $lwaf.lb <1> [list focus $lwaf.lb]
  bind $lwaf.sb <1> [list focus $lwaf.sb]
  bind $lwaf.lb <ButtonRelease> [list takeSelectedGap]
  bind $lwaf.lb <KeyRelease> [list takeSelectedGap]

  pack $lwaf.lb $lwaf.sb -side left -expand "no" -fill "y"

  set rdescur0 "current: "
  set rdescur1 "--"
  label $lwac.lcur0 -text $rdescur0
  set gaplabel [label $lwac.lcur1 -text $rdescur1 -width 3]
  pack  $lwac.lcur0 $lwac.lcur1 -side left


  set sorttype 0

  label $lwas.sort -text "Sorting:"
  grid $lwas.sort -row 0 -column 0 -columnspan 1
  radiobutton $lwas.st0 -text "loaded" -anchor "w" -variable sorttype -value 0 -bd 1 -command takeSortedList
  radiobutton $lwas.st1 -text "e-s-d" -anchor "w" -variable sorttype -value 1 -bd 1 -command takeSortedList
  radiobutton $lwas.st2 -text "d-s-e" -anchor "w" -variable sorttype -value 2 -bd 1 -command takeSortedList
  radiobutton $lwas.st3 -text "s-e-d" -anchor "w" -variable sorttype -value 3 -bd 1 -command takeSortedList
  radiobutton $lwas.st4 -text "s-d-e" -anchor "w" -variable sorttype -value 4 -bd 1 -command takeSortedList
  grid $lwas.st0 -row 0 -column 1 -sticky "we"
  grid $lwas.st1 -row 1 -column 0 -sticky "we"
  grid $lwas.st2 -row 2 -column 0 -sticky "we"
  grid $lwas.st3 -row 1 -column 1 -sticky "we"
  grid $lwas.st4 -row 2 -column 1 -sticky "we"


  label $lwi.l1 -text " " -width 12
  pack $lwi.l1 -side left

  set lwbm [frame $lwb.matrix]
  set lwbs [frame $lwb.fscale]
  pack $lwbm $lwbs -side left

  label $lwbm.ht -text "tuplets\n "
  label $lwbm.h0 -text "exact\nmatch" -justify "left"
  label $lwbm.h1 -text "single\nmism." -justify "left"
  label $lwbm.h2 -text "double\nmism." -justify "left"
  grid $lwbm.ht -column 0 -row 0
  grid $lwbm.h0 -column 1 -row 0
  grid $lwbm.h1 -column 2 -row 0
  grid $lwbm.h2 -column 3 -row 0

  set rowcount $incrowvis
  set rowlist [list]
  for {set ind 1} {$ind <= $rowcount} {incr ind} {
    lappend rowlist $ind
  }

  foreach rind $rowlist {
    global tpl$rind
    set tpl$rind ""
    set cell [entry $lwbm.d-r$rind -textvariable tpl$rind -width 10 -font $fontFixed -bd 1]
    $cell configure -state "readonly" -readonlybackground "#ffffff" -disabledforeground "#000000"
    grid $cell -column 0 -row $rind
    bind $cell <Control-a> [list copyIncidences $cell]
    bind $cell <Control-A> [list copyIncidences $cell]
  }
  foreach cind [list 1 2 3] {
    foreach rind $rowlist {
      global inc$rind-$cind
      set inc$rind-$cind ""
      set cell [entry $lwbm.d-r$rind-c$cind -textvariable inc$rind-$cind -width 6 -font $fontValues -bd 1]
      $cell configure -state "readonly" -readonlybackground "#ffffff" -disabledforeground "#000000"
      grid $cell -column $cind -row $rind
      bind $cell <Control-a> [list copyIncidences $cell]
      bind $cell <Control-A> [list copyIncidences $cell]
    }
  }

  set tuploff 0
  label $lwbs.i0 -text " " -width 2
  set tuploffscale [scale $lwbs.rscale -orient "vertical" -length 200 -from 0 -to 0 -variable tuploff -command setTuplOffset]
  $tuploffscale configure -bigincrement 10
  pack $lwbs.i0 $lwbs.rscale -side left

  bind $lwbs.rscale <1> [list focus $lwbs.rscale]
  bind $lwbs.rscale <Shift-KeyPress> "scrollTups 1 %K"
  bind $lwbs.rscale <KeyPress> "scrollTups 0 %K"

}
prepareIncd $lw

proc scrollTups {shift what} {
  global tuploffscale
  global tuploff

  set toscroll 0
  if {0 == [string compare -nocase "Up" $what]} {
    if {1 == $shift} {
      set toscroll 1
    }
  }
  if {0 == [string compare -nocase "Left" $what]} {
    if {1 == $shift} {
      set toscroll 1
    }
  }
  if {0 == [string compare -nocase "Down" $what]} {
    if {1 == $shift} {
      set toscroll 2
    }
  }
  if {0 == [string compare -nocase "Right" $what]} {
    if {1 == $shift} {
      set toscroll 2
    }
  }
  if {0 == [string compare -nocase "Prior" $what]} {
    set toscroll 3
  }
  if {0 == [string compare -nocase "Next" $what]} {
    set toscroll 4
  }

  if {0 == $toscroll} {
    return
  }

  set sfrom [lindex [split [$tuploffscale cget "-from"] "."] 0]
  set sto [lindex [split [$tuploffscale cget "-to"] "."] 0]

  set aux 0

  if {1 == $toscroll} {
    set aux [expr {$tuploff - 9}]
    if {$sfrom > $aux} {
      set aux $sfrom
    }
    set tuploff $aux
  }
  if {3 == $toscroll} {
    set aux [expr {$tuploff - 100}]
    if {$sfrom > $aux} {
      set aux $sfrom
    }
    set tuploff $aux
  }

  if {2 == $toscroll} {
    set aux [expr {$tuploff + 9}]
    if {$sto < $aux} {
      set aux $sto
    }
    set tuploff $aux
  }
  if {4 == $toscroll} {
    set aux [expr {$tuploff + 100}]
    if {$sto < $aux} {
      set aux $sto
    }
    set tuploff $aux
  }

  setTuplOffset $aux

}


proc setTuplOffset {newValue} {
  global tuploff
  global incusegap

  global inccases
  global inctuplets
  global incrowvis
  global inccurlist
  global incrowlen


  set userows $incrowvis
  if {$userows > $incrowlen} {
    set userows $incrowlen
  }

  for {set rind 1} {$rind <= $incrowvis} {incr rind} {
    global tpl$rind
    foreach cind [list 1 2 3] {
      global inc$rind-$cind
    }
  }

  if {-1 == $incusegap} {
    for {set rind 1} {$rind <= $incrowvis} {incr rind} {
      set tpl$rind ""
      foreach cind [list 1 2 3] {
        set inc$rind-$cind ""
      }
    }
    return
  }

  for {set rind 1} {$rind <= $userows} {incr rind} {
    set curind [lindex $inccurlist [expr {$tuploff + $rind - 1}]]

    set tpl$rind [lindex $inctuplets $curind]

    set curcase [lindex $inccases $curind]
    foreach cind [list 1 2 3] {
      set curcol [expr {(3 * $incusegap) + $cind - 1}]
      set curval [lindex $curcase $curcol]

      set inc$rind-$cind $curval
    }

  }

}

proc takeSortedList {{ind -1}} {
  global incusegap

  global sorttype
  global selgapbox
  global incinilist
  global inccurlist
  global incrowlen

  global sortedesd
  global sorteddse
  global sortedsde
  global sortedsed

  global incgaplist
  global gaplabel

  if {0 == $incrowlen} {
    return
  }

  if {0 == $sorttype} {
    set inccurlist $incinilist
  }

  set gapind $ind
  if {-1 == $ind} {
    set gapind [$selgapbox index active]
    $selgapbox selection clear 0 end
    $selgapbox selection set $gapind
    set gaplen [lindex $incgaplist $gapind]
    $gaplabel configure -text "$gaplen"
  }

  if {1 == $sorttype} {
    set inccurlist [lindex [array get sortedesd $gapind] 1]
  }
  if {2 == $sorttype} {
    set inccurlist [lindex [array get sorteddse $gapind] 1]
  }
  if {3 == $sorttype} {
    set inccurlist [lindex [array get sortedsed $gapind] 1]
  }
  if {4 == $sorttype} {
    set inccurlist [lindex [array get sortedsde $gapind] 1]
  }

  set incusegap $gapind
  setTuplOffset -1
  return

}

proc takeSelectedGap {} {
  global selgapbox
  global incgaplist
  global incrowlen
  global gaplabel
  global incusegap

  if {0 == $incrowlen} {
    return
  }

  set gaplen "--"
  set gaplist [$selgapbox curselection]

  set gapind -1
  set curlen [llength $gaplist]
  if {0 < $curlen} {
    set gapind [lindex $gaplist 0]
    set gaplen [lindex $incgaplist $gapind]
  }
  if {0 == $curlen} {
    set gapind [$selgapbox index active]
    set gaplen [lindex $incgaplist $gapind]
    $selgapbox selection set $gapind
  }
  $gaplabel configure -text "$gaplen"

  if {-1 < $gapind} {
    set incusegap $gapind
    takeSortedList $gapind
  }

}

proc DrawIncd {} {
  global topf
  global pfcurrent
  global lw
  global isAsking
  global fmenu
  global btlist

  global frDefFile
  global topname
  global topw

  if {1 == $isAsking} {
    return
  }

  set nametoset [file tail $frDefFile]
  set strset ""
  if {0 < [string length $nametoset]} {
    set strset "$nametoset - "
  }
  append strset $topname
  wm title $topw $strset

  foreach butt $btlist {
    $fmenu.$butt configure -relief "raised"
  }
  $fmenu.mincd configure -relief "sunken"

  if {"none" != $pfcurrent} {
    pack forget $pfcurrent
  }
  set pfcurrent $lw
  pack $lw -expand yes -fill none -side top

}

#-----------------------------------------

set tupletrank -1
set tupletmatch -1
set tupletpos1 -1
set tupletpos2 -1
set nw [frame $topf.nw]
proc prepareSeqn {nw} {
  global nwb
  global nwbtxt
  global nwbpos

  global selpos1
  global selpos2

  global fontFixed
  global fontValues

  global labexact
  global tupexact
  global tuplet1
  global tupletgap
  global tuplet2

  global tupletrank
  global tupletmatch
  global tupletpos1
  global tupletpos2

  global sequencewrap
  global sequencejoin

  global bfind
  global seekList
  global tupviewstr

  set nwa [frame $nw.a]
  set nwb [frame $nw.b]
  pack $nwa -side top -expand "no" -fill "x"
  pack $nwb -side top -expand "yes" -fill "both"

  set nwal [frame $nwa.l]
  set nwai1 [frame $nwa.i1]
  set nwam [frame $nwa.m]
  set nwai2 [frame $nwa.i2]
  set nwar [frame $nwa.r]
  pack $nwal -side left
  pack $nwai1 -side left -expand "yes" -fill "x"
  pack $nwam -side left
  pack $nwai2 -side left -expand "yes" -fill "x"
  pack $nwar -side right

  label $nwal.li1 -text " "

  set labexact [label $nwal.le -text "exact"]
  $labexact configure -state "disabled"
  set tupexact 0
  checkbutton $nwal.ce -text "" -command exactTuplet -variable tupexact -bd 1
  label $nwal.li2 -text ""

  set tuplet1 "aaa"
  set tupletgap "0"
  set tuplet2 "aaa"

  entry $nwal.et1 -textvariable tuplet1 -width 8 -font $fontFixed -foreground "#f00000" -background "#ffffe8" -bd 1
  entry $nwal.eti -textvariable tupletgap -width 2 -font $fontFixed -foreground "#000000" -background "#ffffe8" -bd 1
  entry $nwal.et2 -textvariable tuplet2 -width 8 -font $fontFixed -foreground "#f00000" -background "#ffffe8" -bd 1
  $nwal.et1 configure -validate "key" -vcmd {keeptuplet1 %P} -highlightthickness 1
  $nwal.eti configure -validate "key" -vcmd {keeptupletgap %P} -highlightthickness 1
  $nwal.et2 configure -validate "key" -vcmd {keeptuplet2 %P} -highlightthickness 1

  set bfind [button $nwal.bf -text "Find" -command tupletLookup -pady 1 -padx 4 -bd 1]
  $bfind configure -state "disabled"

  pack $nwal.li1 $nwal.le $nwal.ce $nwal.li2 $nwal.et1 $nwal.eti $nwal.et2 $nwal.bf -side left

  label $nwai1.l1 -text " "
  pack $nwai1.l1 -side left

  button $nwam.mln -text "<<" -command "tupletSeek - 10" -pady 1 -padx 4 -bd 1
  button $nwam.ml1 -text "<" -command "tupletSeek - 1" -width 2 -pady 1 -padx 4 -bd 1

  set tupviewstr ""
  entry $nwam.cp -width 12 -textvariable tupviewstr -bd 0
  $nwam.cp configure -state "readonly" -highlightthickness 1
  $nwam.cp configure -readonlybackground "#f8f8f8"

  button $nwam.mr1 -text ">" -command "tupletSeek + 1" -width 2 -pady 1 -padx 4 -bd 1
  button $nwam.mrn -text ">>" -command "tupletSeek + 10" -pady 1 -padx 4 -bd 1

  set seekList [list $nwam.mln $nwam.ml1 $nwam.mrn $nwam.mr1]
  foreach wdg $seekList {
    $wdg configure -state "disabled"
  }

  pack $nwam.mrn $nwam.mr1 $nwam.cp \
    $nwam.ml1 $nwam.mln -side right

  label $nwai2.l1 -text " "
  pack $nwai2.l1 -side left

  set sequencewrap 0
  set sequencejoin 0

  checkbutton $nwar.jc -text "join" -variable sequencejoin -command joinSeqs -bd 1
  label $nwar.ji -text ""
  checkbutton $nwar.wc -text "wrap" -variable sequencewrap -command wrapSeqs -bd 1
  label $nwar.wi -text " "

  pack $nwar.wi $nwar.wc $nwar.ji $nwar.jc -side right

  set nwbtxt [text $nwb.txt -height 20 -width 60 -xscrollcommand "seqScaleSet" -yscrollcommand "$nwb.scy set" -bd 1]
  $nwbtxt configure -font $fontFixed
  $nwbtxt configure -exportselection 1
  $nwbtxt configure -selectbackground "#c0c080"

  $nwbtxt configure -background "#ffffff"
  $nwbtxt configure -wrap "none" -state "disabled"

  set nwbpos [text $nwb.pos -height 1 -width 60 -bd 0]
  $nwbpos configure -font $fontFixed
  $nwbpos configure -exportselection 1
  $nwbpos configure -selectbackground "#c0c080"
  $nwbpos configure -background "#ffffff"
  $nwbpos configure -wrap "none" -state "disabled"

  set nwbscx [scrollbar $nwb.scx -bd 1 -orient "horizontal"]
  set nwbscy [scrollbar $nwb.scy -bd 1]
  $nwbscx configure -command moveTxt
  $nwbscy configure -command moveTyt

  grid $nwbtxt -row 0 -column 0 -sticky "snew"
  grid $nwbpos -row 1 -column 0 -sticky "we"
  grid $nwbscx -row 2 -column 0 -sticky "we"
  grid $nwbscy -row 0 -column 1 -sticky "sn"

  grid rowconfigure $nwb 0 -weight 1
  grid columnconfigure $nwb 0 -weight 1

  bind $nwbtxt <1> [list focus $nwbtxt]
  bind $nwbscx <1> [list focus $nwbscx]
  bind $nwbscy <1> [list focus $nwbscy]
  bind $nwbpos <1> [list focus $nwbscx]

  bind $nwbtxt <Control-s> selectedLookup
  bind $nwbtxt <Control-S> selectedLookup
  bind $nwbtxt <Control-r> [list ::tk_textCopy $nwbtxt 1 0]
  bind $nwbtxt <Control-R> [list ::tk_textCopy $nwbtxt 1 0]
  bind $nwbtxt <Control-g> [list ::tk_textCopy $nwbtxt 1 1]
  bind $nwbtxt <Control-G> [list ::tk_textCopy $nwbtxt 1 1]

}
prepareSeqn $nw

proc seqScaleSet {{par1 ""} {par2 ""} {par3 ""} {par4 ""}} {
  global nwb

  if {"" == $par1} {
    $nwb.scx set
  } elseif {"" == $par2} {
    $nwb.scx set $par1
  } elseif {"" == $par3} {
    $nwb.scx set $par1 $par2
  } elseif {"" == $par4} {
    $nwb.scx set $par1 $par2 $par3
  } else {
    $nwb.scx set $par1 $par2 $par3 $par4
  }

  updateGauge
}

proc updateGauge {} {
  global nwbtxt
  global nwbpos

  global gmove
  global gxtxt
  global gxpos

  if {0 == $gmove} {
    return
  }

  set posfrc 0.0
  set txtfrc [lindex [$nwbtxt xview] 0]
  if {0.0 < $txtfrc} {
    set posfrc [expr {((1.0 * $txtfrc) * $gxtxt) / $gxpos}]
  }
  $nwbpos xview moveto $posfrc

}

proc moveTxt {what pos {how "none"}} {
  global nwbtxt
  global nwbpos

  global gmove
  global gxtxt
  global gxpos

  if {"none" == $how} {
    $nwbtxt xview $what $pos
  } else {
    $nwbtxt xview $what $pos $how
  }

  updateGauge
}

proc moveTyt {what pos {how "none"}} {
  global nwbtxt
  global nwbpos

  global gmove
  global gxtxt
  global gxpos

  if {"none" == $how} {
    $nwbtxt yview $what $pos
  } else {
    $nwbtxt yview $what $pos $how
  }

  updateGauge
}

set gmove 1
set gxtxt 0
set gxpos 0
set gotpx 0
proc setGPixs {} {
  global nwbtxt
  global nwbpos

  global totLines
  global sequencewrap
  global gotpx

  global gmove
  global gxtxt
  global gxpos
  global gotpx

  set gxtxt 0
  set gmove 0
  if {0 == $totLines} {
    set gotpx 1
    return
  }

  if {1 == $sequencewrap} {
    return
  }
  set gotpx 1

  for {set ind 1} {$ind <= $totLines} {incr ind 5} {
    set curinfo [$nwbtxt dlineinfo "$ind.0"]
    if {0 < [llength $curinfo]} {
      set curlen [lindex $curinfo 2]
      if {$gxtxt < $curlen} {
        set gxtxt $curlen
      }
    }
  }

  set gmove 1

}

proc makeGauge {glen} {
  global nwbpos
  global gxpos

  $nwbpos configure -state "normal"
  $nwbpos delete 1.0 end

  set cnt 7
  if {$cnt < $glen} {
    set cnt $glen
  }
  set cmax 1000000
  if {$cmax < $cnt} {
    set cnt $cmax
  }

  for {set ind 0} {$ind < $cnt} {incr ind} {
    set cnum [expr {(10 * $ind) + 1}]
    set clen [string length $cnum]
    $nwbpos insert end $cnum
    set cend [string repeat " " [expr 10 - $clen]]
    $nwbpos insert end $cend
  }

  set lend [expr {600 - ($cnt * 10)}]
  set send [string repeat " " $lend]
  $nwbpos insert end $send

  $nwbpos configure -state "disabled"

  set gxpos [lindex [$nwbpos dlineinfo 1.0] 2]
  $nwbpos xview moveto 0.0

}
makeGauge 0

set sqseen 0
set sqlate 0
proc DrawSeqn {} {
  global topf
  global pfcurrent
  global nw
  global isAsking
  global fmenu
  global btlist

  global gotpx
  global sqseen
  global sqlate
  global sequencewrap

  global sqDefFile
  global topname
  global topw

  if {1 == $isAsking} {
    return
  }

  set nametoset [file tail $sqDefFile]
  set strset ""
  if {0 < [string length $nametoset]} {
    set strset "$nametoset - "
  }
  append strset $topname
  wm title $topw $strset

  foreach butt $btlist {
    $fmenu.$butt configure -relief "raised"
  }
  $fmenu.mseqn configure -relief "sunken"

  if {"none" != $pfcurrent} {
    pack forget $pfcurrent
  }
  set pfcurrent $nw
  pack $nw -expand "yes" -fill "both" -side top

  set sqseen 1
  if {0 != $sqlate} {
    update

    makeGauge $sqlate
    if {0 == $sequencewrap} {
      updateGauge
    }
    setGPixs

    set sqlate 0
  }

}

proc keeptuplets {val} {

  if {20 < [string length $val]} {
    return 0
  }

  if {1 == [string is alpha $val]} {
    if {1 == [string is ascii $val]} {
      return 1
    }
  }

  return 0
}

proc keeptuplet1 {val} {
  global tuplet2

  set rv [keeptuplets $val]

  if {1 == $rv} {
    updateBFind [list $val $tuplet2]
  }

  return $rv

}
proc keeptuplet2 {val} {
  global tuplet1

  set rv [keeptuplets $val]

  if {1 == $rv} {
    updateBFind [list $tuplet1 $val]
  }

  return $rv

}

proc keeptupletgap {val} {
  if {2 < [string length $val]} {
    return 0
  }

  if {1 == [string is digit $val]} {
    if {1 == [string is ascii $val]} {
      return 1
    }
  }

  return 0
}

proc exactTuplet {} {
  global tupexact
  global labexact

  if {1 == $tupexact} {
    $labexact configure -state "normal"
  } else {
    $labexact configure -state "disabled"
  }

  updateBFind

}

proc wrapSeqs {} {
  global sequencewrap
  global nwbtxt
  global nwbpos

  global gotpx

  $nwbtxt configure -state "normal"
  if {1 == $sequencewrap} {
    $nwbtxt configure -wrap "char"
  } else {
    $nwbtxt configure -wrap "none"
  }
  $nwbtxt configure -state "disabled"

  $nwbpos xview moveto 0.0

  if {0 == $gotpx} {
    setGPixs
  }

  dispSeqs

}

set totLines 0
proc dispSeqs {} {
  global sequencejoin
  global sequencewrap
  global nwbtxt

  global fontValues
  global fontLesser

  global seqheads
  global seqlines

  global totLines
  global gotpx
  global sqlate

  global tupscur

  set gotpx 0

  set scount [llength $seqheads]

  $nwbtxt configure -state "normal"
  $nwbtxt delete 1.0 end

  set maxlen 0
  if {0 == $sequencewrap} {
    for {set sind 0} {$sind < $scount} {incr sind} {
      set curline [lindex $seqheads $sind]
      set curlen [string length $curline]
      if {$maxlen < $curlen} {
        set maxlen $curlen
      }

      set curlist [lindex $seqlines $sind]
      if {1 == $sequencejoin} {
        set curlist [list [join $curlist ""]]
      }

      foreach line $curlist {
        set curlen [string length $line]
        if {$maxlen < $curlen} {
          set maxlen $curlen
        }

      }
    }
  }

  set tagfgeval "$nwbtxt tag add grayfg"
  set toevaltagfg 0
  set tagbgeval "$nwbtxt tag add graybg"
  set toevaltagbg 0

  set rind 0
  for {set sind 0} {$sind < $scount} {incr sind} {
    incr rind
    set curline [lindex $seqheads $sind]
    if {0 == $sequencewrap} {
      set curlen [string length $curline]
      append curline [string repeat " " [expr {$maxlen - $curlen}]]
    }
    $nwbtxt insert end $curline
    $nwbtxt insert end "\n"

    append tagfgeval " $rind.0 $rind.end"
    set toevaltagfg 1

    set curlist [lindex $seqlines $sind]
    if {1 == $sequencejoin} {
      set curlist [list [join $curlist ""]]
    }

    set isempty 0
    if {0 == [llength $curlist]} {
      set isempty 1
    }
    if {1 == [llength $curlist]} {
      if {0 == [string length [lindex $curlist 0]]} {
        set isempty 1
      }
    }

    if {1 == $isempty} {
      incr rind
      if {0 == $sequencewrap} {
        $nwbtxt insert end [string repeat " " $maxlen]
      }
      $nwbtxt insert end "\n"
      continue
    }

    set globeven 0
    foreach line $curlist {
      set globeven [expr {1 - $globeven}]
      incr rind
      set curline $line
      set curlen [string length $line]
      if {0 == $sequencewrap} {
        append curline [string repeat " " [expr {$maxlen - $curlen}]]
      }
      $nwbtxt insert end $curline
      $nwbtxt insert end "\n"

      set even $globeven
      set ccount [expr {[string length $line] / 10}]
      if {0 < [expr {[string length $line] % 10}]} {
        incr ccount
      }
      for {set cind 0} {$cind < $ccount} {incr cind} {
        set even [expr {1 - $even}]
        if {1 == $even} {
          set cstart [expr {10 * $cind}]
          set cend [expr {10 * (1 + $cind)}]
          if {$cend > $curlen} {
            set cend $curlen
          }
          append tagbgeval " $rind.$cstart $rind.$cend"
          set toevaltagbg 1
        }
      }
    }

  }

  if {1 == $toevaltagfg} {
    eval $tagfgeval
    $nwbtxt tag configure "grayfg" -foreground "#60a0a0"
  }
  if {1 == $toevaltagbg} {
    eval $tagbgeval
    $nwbtxt tag configure "graybg" -background "#f4f4f8"
  }

  $nwbtxt tag raise sel
  $nwbtxt configure -state "disabled"

  set totLines $rind

  if {0 == $sqlate} {
    setGPixs
  }

  if {0 == $sequencewrap} {
    if {0 == $sqlate} {
      updateGauge
    }
  }

  setLookups
  tupletSeek "=" $tupscur

  return
}

proc joinSeqs {} {

  dispSeqs
}

proc updateBFind {{newval "-"}} {
  global bfind
  global totLines
  global tupexact

  global tuplet1
  global tuplet2

  if {0 == $totLines} {
    return
  }

  set val1 ""
  set val2 ""

  if {"-" == $newval} {
    set val1 $tuplet1
    set val2 $tuplet2
  } else {
    set val1 [lindex $newval 0]
    set val2 [lindex $newval 1]
  }

  set len1 [string length $val1]
  set len2 [string length $val2]

  if {1 == $tupexact} {
    if {0 < $len1} {
      $bfind configure -state "normal"
    } else {
      $bfind configure -state "disabled"
    }
    return
  }

  if {2 > $len1} {
    $bfind configure -state "disabled"
    return
  }

  $bfind configure -state "normal"
  return

}

set totFound 0
proc updateBSeek {} {
  global totFound
  global seekList

  set nstate "normal"
  if {0 == $totFound} {
    set nstate "disabled"
  }
  foreach wdg $seekList {
    $wdg configure -state $nstate
  }

}

set tupscur 0
set conseeklist [list]
set sepseeklist [list]
set setviewtags [list]
proc tupletSeek {dir pos} {
  global totFound
  global rnkFound
  global sequencejoin
  global nwbtxt

  global tupscur
  global conseeklist
  global sepseeklist
  global setviewtags

  global tupviewstr
  global seqstarts

  if {0 == $totFound} {
    return
  }

  removeViewTags

  if {"+" == $dir} {
    set tupscur [expr {$tupscur + $pos}]
  }
  if {"-" == $dir} {
    set tupscur [expr {$tupscur - $pos}]
  }
  if {"=" == $dir} {
    set tupscur $pos
  }

  if {0 > $tupscur} {
    set tupscur 0
  }
  if {$tupscur >= $totFound} {
    set tupscur [expr {$totFound - 1}]
  }

  set auxpair [lindex $rnkFound $tupscur]
  set iline [lindex $auxpair 0]
  set ipost [lindex $auxpair 1]

  set curori [lindex $seqstarts $iline]
  set abspos ""
  if {"" != $curori} {
    set abspos [expr {$curori + $ipost}]
    if {0 > $curori} {
      if {0 <= $abspos} {
        incr abspos
      }
    }
  }

  incr iline
  incr ipost
  set tupviewstr " $iline: $ipost"

  if {"" != $curori} {
    append tupviewstr " (" $abspos ")"
  }

  set selcols [list "#ffff80" "#a0f0c0"]

  set toseeindx [list]

  set setviewtags [list]
  set curind 0
  if {1 == $sequencejoin} {

    set viewtags [lindex $conseeklist $tupscur]
    foreach onetagpair $viewtags {
      incr curind
      set tname tview$curind
      set colind [lindex $onetagpair 0]
      set colcur [lindex $selcols $colind]
      set curtag [lindex $onetagpair 1]
      set curstart [lindex $curtag 0]
      set curstop [lindex $curtag 1]
      $nwbtxt tag add $tname $curstart $curstop
      $nwbtxt tag configure $tname -underline 1
      $nwbtxt tag configure $tname -background $colcur
      lappend setviewtags $tname
      lappend toseeindx $curstart $curstop
    }

  } else {

    set viewtaglists [lindex $sepseeklist $tupscur]
    foreach onetaglistpair $viewtaglists {
      set colind [lindex $onetaglistpair 0]
      set colcur [lindex $selcols $colind]
      foreach onetag [lindex $onetaglistpair 1] {
        incr curind
        set tname tview$curind
        set curstart [lindex $onetag 0]
        set curstop [lindex $onetag 1]
        $nwbtxt tag add $tname $curstart $curstop
        $nwbtxt tag configure $tname -underline 1
        $nwbtxt tag configure $tname -background $colcur
        lappend setviewtags $tname
        lappend toseeindx $curstart $curstop
      }
    }

  }

  $nwbtxt tag raise sel

  set tslen [llength $toseeindx]
  incr tslen -1
  for {set tsind $tslen} {$tsind >= 0} {incr tsind -1} {
    set curindex [lindex $toseeindx $tsind]
    $nwbtxt see $curindex
  }

  updateGauge

}

proc selectedLookup {} {
  global sequencejoin
  global nwbtxt

  global tupStartsCon
  global tupStartsSep

  set curlist [list]
  if {1 == $sequencejoin} {
    set curlist $tupStartsCon
  } else {
    set curlist $tupStartsSep
  }
  if {0 == [llength $curlist]} {
    return
  }

  set selranges [$nwbtxt tag ranges sel]
  if {0 == [llength $selranges]} {
    return
  }

  set onesel [lindex $selranges 0]
  set onesellist [split $onesel "."]
  set oneselrow [lindex $onesellist 0]
  set oneselcol [lindex $onesellist 1]

  set newind -1

  set curind -1
  foreach oneind $curlist {
    incr curind
    set oneindlist [split $oneind "."]
    set oneindrow [lindex $oneindlist 0]
    set oneindcol [lindex $oneindlist 1]

    if {$oneindrow < $oneselrow} {
      continue
    }
    if {$oneindrow == $oneselrow} {
      if {$oneindcol < $oneselcol} {
        continue
      }
    }

    set newind $curind
    break

  }

  if {-1 < $newind} {
    tupletSeek "=" $newind
  }

}

proc removeViewTags {} {
  global tupviewstr
  global setviewtags
  global nwbtxt

  set tupviewstr ""

  foreach vtag $setviewtags {
    $nwbtxt tag delete $vtag
  }

  set setviewtags [list]

}

set lookupTags0 [list]
set lookupTags1 [list]
proc deleteLookups {} {
  global totFound
  global posFound
  global rnkFound
  global nwbtxt
  global lookupTags0
  global lookupTags1

  global tagConBorders0
  global tagSepBorders0
  global tagConBorders1
  global tagSepBorders1

  global tupscur
  global conseeklist
  global sepseeklist

  global tupviewstr

  set tupviewstr ""

  set tagConBorders0 [list]
  set tagSepBorders0 [list]
  set tagConBorders1 [list]
  set tagSepBorders1 [list]

  foreach curtag $lookupTags0 {
    $nwbtxt tag delete $curtag
  }
  foreach curtag $lookupTags1 {
    $nwbtxt tag delete $curtag
  }

  set totFound 0
  set posFound [list]
  set rnkFound [list]

  removeViewTags
  set tupscur 0

}

set tupStartsCon [list]
set tupStartsSep [list]
set tagConBorders0 [list]
set tagSepBorders0 [list]
set tagConBorders1 [list]
set tagSepBorders1 [list]
proc prepareLookups {tpseqs} {
  global totFound
  global posFound
  global rnkFound
  global seqlines

  global tuplet1
  global tuplet2

  global tagConBorders0
  global tagSepBorders0
  global tagConBorders1
  global tagSepBorders1

  global conseeklist
  global sepseeklist
  global tupviewstr

  global tupStartsCon
  global tupStartsSep

  set tagConBorders0 [list]
  set tagSepBorders0 [list]
  set tagConBorders1 [list]
  set tagSepBorders1 [list]

  set tupStartsCon [list]
  set tupStartsSep [list]

  set conseeklist [list]
  set sepseeklist [list]
  set tupviewstr ""
  if {0 == $totFound} {
    set posFound [list]
    set rnkFound [list]
    return
  }

  set saddlist [list 0]
  set saddcur 0
  foreach slist $seqlines {
    set curlen [llength $slist]
    set addlen [expr {$curlen - 1}]
    if {0 < $addlen} {
      incr saddcur $addlen
      lappend saddlist $saddcur
    }
  }

  set tuplist [list $tuplet1 $tuplet2]
  set tupcomb [llength $tpseqs]

  set rind 0
  set sind 0
  foreach sqlist $posFound {
    incr rind 2
    set clines [lindex $seqlines $sind]
    set cbases [list]
    foreach oneline $clines {
      lappend cbases [string length $oneline]
    }
    set cseq [join $clines ""]
    set rbase [lindex $saddlist $sind]

    foreach sqpos $sqlist {
      set curseeklistcon [list]
      set curseeklistsep [list]
      for {set pind 0} {$pind < $tupcomb} {incr pind} {
        set tuplet [lindex $tuplist $pind]
        set pospair [lindex $tpseqs $pind]
        set aux0 [lindex $pospair 0]
        set aux1 [lindex $pospair 1]
        set cstart [expr {$sqpos + $aux0}]
        set cstop [expr {$sqpos + $aux1 - 1}]

        set onetag [list "$rind.$cstart" "$rind.[expr {$cstop + 1}]"]
        if {0 == $pind} {
          lappend tupStartsCon "$rind.$cstart"
        }

        set onetaglist [list]
        set curbase 0
        set bind 0
        foreach onebase $cbases {
          if {[expr {$onebase + $curbase}] <= $cstart} {
            incr curbase $onebase
            incr bind
          } else {
            break
          }
        }
        set seprow [expr {$rbase + $rind + $bind}]
        set sepstart $cstart
        set sepstop $cstop
        for {set sepind 0} {$sepind < $bind} {incr sepind} {
          set auxlen [string length [lindex $clines $sepind]]
          set sepstart [expr {$sepstart - $auxlen}]
          set sepstop [expr {$sepstop - $auxlen}]
        }
        for {set sepind $bind} {$sepind < [llength $clines]} {incr sepind} {
          set auxlen [string length [lindex $clines $sepind]]
          if {0 == $pind} {
            if {$bind == $sepind} {
              lappend tupStartsSep "$seprow.$sepstart"
            }
          }
          if {$auxlen > $sepstop} {
            lappend onetaglist [list "$seprow.$sepstart" "$seprow.[expr {$sepstop + 1}]"]
            break
          }
          lappend onetaglist [list "$seprow.$sepstart" "$seprow.$auxlen"]
          incr seprow
          set sepstart 0
          set sepstop [expr {$sepstop - ($auxlen - $sepstart)}]
        }

        set sfound [string range $cseq $cstart $cstop]
        set tdiff [string compare $tuplet $sfound]
        set mmcount 0
        if {0 == $tdiff} {
          lappend tagConBorders0 $onetag
          lappend tagSepBorders0 $onetaglist
        } else {
          lappend tagConBorders1 $onetag
          lappend tagSepBorders1 $onetaglist
          set mmcount 1
        }

        lappend curseeklistcon [list $mmcount $onetag]
        lappend curseeklistsep [list $mmcount $onetaglist]
      }

      lappend conseeklist $curseeklistcon
      lappend sepseeklist $curseeklistsep
    }

    incr sind
  }


  tupletSeek "=" 0

}

proc setLookups {} {
  global tagConBorders0
  global tagSepBorders0
  global tagConBorders1
  global tagSepBorders1

  global lookupTags0
  global lookupTags1

  global nwbtxt
  global sequencejoin

  set lookupTags0 [list]
  set lookupTags1 [list]

  set tagdarkeval "$nwbtxt tag add ludark"
  set toevaltagdark 0
  set taglighteval "$nwbtxt tag add lulight"
  set toevaltaglight 0

  $nwbtxt configure -state "normal"

  if {1 == $sequencejoin} {

    foreach onetag $tagConBorders1 {
      append tagdarkeval " [lindex $onetag 0] [lindex $onetag 1]"
      set toevaltagdark 1
    }
    foreach onetag $tagConBorders0 {
      append taglighteval " [lindex $onetag 0] [lindex $onetag 1]"
      set toevaltaglight 1
    }

  } else {

    set tind 0
    foreach onetaglist $tagSepBorders1 {
      foreach onetag $onetaglist {
        append tagdarkeval " [lindex $onetag 0] [lindex $onetag 1]"
        set toevaltagdark 1
      }
    }
    set tind 0
    foreach onetaglist $tagSepBorders0 {
      foreach onetag $onetaglist {
        append taglighteval " [lindex $onetag 0] [lindex $onetag 1]"
        set toevaltaglight 1
      }
    }

  }

  if {1 == $toevaltagdark} {
    eval $tagdarkeval
    $nwbtxt tag configure "ludark" -background "#c8a8a8"
    lappend lookupTags1 "ludark"
  }
  if {1 == $toevaltaglight} {
    eval $taglighteval
    $nwbtxt tag configure "lulight" -background "#ffa0a0"
    lappend lookupTags1 "lulight"
  }

  $nwbtxt tag raise sel
  $nwbtxt configure -state "disabled"

}

set posFound [list]
set rnkFound [list]
proc tupletLookup {} {
  global totLines
  global totFound
  global posFound
  global rnkFound
  global tupStartsCon
  global tupStartsSep

  global sequencewrap
  global seqheads
  global seqlines

  global tuplet1
  global tupletgap
  global tuplet2
  global tupexact
  global tupscur

  set tupStartsCon [list]
  set tupStartsSep [list]
  deleteLookups

  updateBSeek
  if {0 == $totLines} {
    return
  }

  set scount [llength $seqheads]

  if {"" == $tupletgap} {
    set tupletgap 0
  }
  if {2 == [string length $tupletgap]} {
    set glist [split $tupletgap ""]
    if {"0" == [lindex $glist 0]} {
      set tupletgap [lindex $glist 1]
    }
  }


  set reparts [list]
  foreach tcur [list $tuplet1 $tuplet2] {
    set reaux [list]
    set tlist [split $tcur ""]
    set tlen [llength $tlist]
    for {set wind 0} {$wind < $tlen} {incr wind} {
      set auxlist $tlist
      lset auxlist $wind "."
      set auxstr [join $auxlist ""]
      lappend reaux $auxstr
    }
    lappend reparts [join $reaux "|"]
  }

  set usegap 1
  if {0 == [string length $tuplet1]} {
    set usegap 0
  }
  if {0 == [string length $tuplet2]} {
    set usegap 0
  }

  set recur ""
  if {0 < [string length $tuplet1]} {
    append recur "(?:"
    append recur [lindex $reparts 0]
    append recur ")"
  }

  if {1 == $usegap} {
    append recur ".\{$tupletgap\}"
  } else {
    set tupletgap 0
  }

  if {0 < [string length $tuplet2]} {
    append recur "(?="
    append recur [lindex $reparts 1]
    append recur ")"
  }

  set rexact "(?:"
  append rexact $tuplet1
  append rexact ")"
  if {1 == $usegap} {
    append rexact ".\{$tupletgap\}"
  }
  if {0 < [string length $tuplet2]} {
    append rexact "(?="
    append rexact $tuplet2
    append rexact ")"
  }

  set restr ""
  if {1 == $tupexact} {
    set restr $rexact
  } else {
    set restr $recur
  }

  set tpseqs [list]
  set tlen1 [string length $tuplet1]
  set tlen2 [string length $tuplet2]
  lappend tpseqs [list 0 $tlen1]
  set toadd2 0
  if {0 < $tlen2} {
    set toadd2 1
  }
  if {1 == $toadd2} {
    set cpstart [expr {$tupletgap + $tlen1}]
    set cpstop [expr {$cpstart + $tlen2}]
    lappend tpseqs [list $cpstart $cpstop]
  }


  for {set sind 0} {$sind < $scount} {incr sind} {
    set seqFound [list]

    set curlist [lindex $seqlines $sind]
    set curlseq [join $curlist ""]

    set tonext 1
    set cfound 0
    set crlist [list]
    set curpos 0
    while {1 == $tonext} {
      set res [regexp -inline -indices -start $curpos $restr $curlseq]
      if {0 != [llength $res]} {
        lappend crlist $res
        set auxpair [lindex $res 0]
        set curpos [expr {1 + [lindex $auxpair 0]}]
        incr totFound
        lappend seqFound [expr {$curpos - 1}]
        lappend rnkFound [list $sind [expr {$curpos - 1}]]
      } else {
        set tonext 0
      }
    }

    lappend posFound $seqFound

  }

  prepareLookups $tpseqs
  setLookups
  tupletSeek "=" $tupscur
  updateBSeek

}

#-----------------------------------------

set aw [frame $topf.aw]
proc prepareAbout {aw} {
  global fontLarger
  global stversion

  set awa [frame $aw.a]
  set awb [frame $aw.b]
  set awc [frame $aw.c]
  pack $awa $awb $awc -side top

  label $awa.lt1 -text " StuDNA results survey " -font $fontLarger -relief raised -bd 1
  text $awa.lt2 -height 1 -width 26 -bd 0 -font [list "Courier" 9] -background "#ffffff"
  $awa.lt2 insert end "http://www.bioplexity.org/"
  $awa.lt2 configure -state disabled
  label $awa.lt3 -text " \nversion $stversion \n " -font [list "Courier" 9]
  label $awa.i1 -text " " -height 4

  pack $awa.lt1 -side top -expand "yes" -fill "y"
  pack $awa.lt2 $awa.lt3 $awa.i1 -side top

  set awbl [frame $awb.l]
  set awbi [frame $awb.i]
  set awbr [frame $awb.r]
  pack $awbl $awbi $awbr -side left

  label $awbl.ld1 -text "Data files" -anchor "w"
  label $awbl.ld2 -text "   studna data load, quit" -anchor "w"
  label $awbl.le1 -text "Estimates" -anchor "w"
  label $awbl.le2 -text "   frequency probabilities" -anchor "w"
  label $awbl.li1 -text "Incidences" -anchor "w"
  label $awbl.li2 -text "   tuplet-gap-tuplet counts" -anchor "w"
  label $awbl.ls1 -text "Sequences" -anchor "w"
  label $awbl.ls2 -text "   tuplet positions look-up" -anchor "w"
  pack $awbl.ld1 $awbl.ld2 $awbl.le1 $awbl.le2 \
    $awbl.li1 $awbl.li2 $awbl.ls1 $awbl.ls2 \
    -side top -expand "no" -fill "both"

  label $awbi.i1 -text " " -width 8
  pack $awbi.i1 -side left

  label $awbr.l0 -text "1) "
  label $awbr.l1 -text "load StuDNA blueprint files"
  label $awbr.l2 -text "and (input) sequences"
  label $awbr.c0 -text "2) "
  label $awbr.c1 -text "choose interesting tuplets"
  label $awbr.c2 -text "- via the matrix parts"
  label $awbr.s0 -text "3) "
  label $awbr.s1 -text "look up selected tuplets"
  label $awbr.s2 -text "- via the sequences part"

  grid $awbr.l0 -row 0 -column 0 -sticky "e"
  grid $awbr.l1 -row 0 -column 1 -sticky "w"
  grid $awbr.l2 -row 1 -column 1 -sticky "w"
  grid $awbr.c0 -row 2 -column 0 -sticky "e"
  grid $awbr.c1 -row 2 -column 1 -sticky "w"
  grid $awbr.c2 -row 3 -column 1 -sticky "w"
  grid $awbr.s0 -row 4 -column 0 -sticky "e"
  grid $awbr.s1 -row 4 -column 1 -sticky "w"
  grid $awbr.s2 -row 5 -column 1 -sticky "w"


  label $awc.li1 -text " \n "
  pack $awc.li1 -side top

}
prepareAbout $aw


proc DrawAbout {} {
  global topf
  global pfcurrent
  global aw
  global isAsking
  global fmenu
  global btlist

  global topname
  global topw

  if {1 == $isAsking} {
    return
  }

  wm title $topw $topname

  foreach butt $btlist {
    $fmenu.$butt configure -relief "raised"
  }
  $fmenu.mhelp configure -relief "sunken"

  if {"none" != $pfcurrent} {
    pack forget $pfcurrent
  }
  set pfcurrent $aw
  pack $aw -expand yes -fill none -side top

}

#-----------------------------------------

set frDefFile ""
set sqDefFile ""

set ww [frame $topf.ww]
proc prepareFile {ww} {
  global loaddir
  global ftypespr
  global ftypesfr
  global ftypesfa

  global prDefFile
  global frDefFile
  global sqDefFile
  global wprwbotf
  global wfrwbotf
  global wsqwbotf

  global wweb0 wwel0
  global wwel wwebs

  global wwi1l1
  global wwi2l1
  global wwi3l1

  set wwpr [frame $ww.pr]
  set wwi1 [frame $ww.i1]
  set wwfr [frame $ww.fr]
  set wwi2 [frame $ww.i2]
  set wwsq [frame $ww.sq]
  set wwi3 [frame $ww.i3]
  set wwe [frame $ww.ex]
  pack $wwpr $wwi1 $wwfr $wwi2 $wwsq -side top
  pack $wwi3 -side top -expand "yes" -fill "y"
  pack $wwe -side top

  set ftypespr {
    {{CSV files} {.csv}}
    {{CSV files} {.txt}}
    {{All files} *}
  }

  set ftypesfr {
    {{CSV files} {.csv}}
    {{CSV files} {.txt}}
    {{All files} *}
  }

  set ftypesfa {
    {{Fasta files} {.fa}}
    {{Fasta files} {.fasta}}
    {{All files} *}
  }

  set wprwboti [frame $wwpr.in]
  set wprwbotf [frame $wwpr.ld]
  pack $wprwboti $wprwbotf -side top

  label $wprwboti.l1 -text "          StuDNA probabilities matrix"
  pack $wprwboti.l1 -side top

  button $wprwbotf.bfload -text " Load " -bd 1 -command {
    set prdefaux [tk_getOpenFile -initialdir $loaddir -filetypes $ftypespr -title "Load count estimations file"]
    if {0 < [string length $prdefaux]} {
      $wprwbotf.ename configure -state "normal"
      set prDefFile $prdefaux
      $wprwbotf.ename configure -state "readonly"
      set loaddir [file dirname $prdefaux]
      set rv [prLoad]
      if {0 == $rv} {
        $wprwbotf.ename configure -state "normal"
        set prDefFile ""
        $wprwbotf.ename configure -state "readonly"
      }
    }
  }
  entry $wprwbotf.ename -textvariable prDefFile -width 60 -bd 1 -relief "sunken"
  $wprwbotf.ename configure -state "readonly"
  $wprwbotf.ename configure -readonlybackground "#ffffff"
  pack $wprwbotf.bfload $wprwbotf.ename -side left

  set wwi1l1 [label $wwi1.l1 -text " " -height 1 -width 20]
  set wwi1l2 [label $wwi1.l2 -text " " -height 1]
  pack $wwi1l1 $wwi1l2 -side top

  set wfrwboti [frame $wwfr.in]
  set wfrwbotf [frame $wwfr.ld]
  pack $wfrwboti $wfrwbotf -side top

  label $wfrwboti.l1 -text "          StuDNA frequencies matrix"
  pack $wfrwboti.l1 -side top

  button $wfrwbotf.bfload -text " Load " -bd 1 -command {
    set frdefaux [tk_getOpenFile -initialdir $loaddir -filetypes $ftypesfr -title "Load tuplet incidences file"]
    if {0 < [string length $frdefaux]} {
      $wfrwbotf.ename configure -state "normal"
      set frDefFile $frdefaux
      $wfrwbotf.ename configure -state "readonly"
      set loaddir [file dirname $frdefaux]
      set rv [frLoad]
      if {0 == $rv} {
        $wfrwbotf.ename configure -state "normal"
        set frDefFile ""
        $wfrwbotf.ename configure -state "readonly"
      }
    }
  }
  entry $wfrwbotf.ename -textvariable frDefFile -width 60 -bd 1 -relief "sunken"
  $wfrwbotf.ename configure -state "readonly"
  $wfrwbotf.ename configure -readonlybackground "#ffffff"
  pack $wfrwbotf.bfload $wfrwbotf.ename -side left

  set wwi2l1 [label $wwi2.l1 -text " " -height 1 -width 20]
  set wwi2l2 [label $wwi2.l2 -text " " -height 1]
  pack $wwi2l1 $wwi2l2 -side top

  set wsqwboti [frame $wwsq.in]
  set wsqwbotf [frame $wwsq.ld]
  pack $wsqwboti $wsqwbotf -side top

  label $wsqwboti.l1 -text "          Fasta formatted sequences"
  pack $wsqwboti.l1 -side top

  button $wsqwbotf.bfload -text " Load " -bd 1 -command {
    set sqdefaux [tk_getOpenFile -initialdir $loaddir -filetypes $ftypesfa -title "Load fasta sequences file"]
    if {0 < [string length $sqdefaux]} {
      $wsqwbotf.ename configure -state "normal"
      set sqDefFile $sqdefaux
      $wsqwbotf.ename configure -state "readonly"
      set loaddir [file dirname $sqdefaux]
      set rv [sqLoad]
      if {0 == $rv} {
        $wsqwbotf.ename configure -state "normal"
        set sqDefFile ""
        $wsqwbotf.ename configure -state "readonly"
      }

    }
  }
  entry $wsqwbotf.ename -textvariable sqDefFile -width 60 -bd 1 -relief "sunken"
  $wsqwbotf.ename configure -state "readonly"
  $wsqwbotf.ename configure -readonlybackground "#ffffff"
  pack $wsqwbotf.bfload $wsqwbotf.ename -side left

  set wwi3l1 [label $wwi3.l1 -text " " -height 1 -width 20]
  set wwi3l2 [label $wwi3.l2 -text " " -height 5]
  pack $wwi3l1 $wwi3l2 -side top

  set wweb0 [button $wwe.b0 -text "Exit" -width 4 -padx 8 -bd 1 -command exQuest]
  set wwel0 [label $wwe.l0 -text " " -height 2]
  pack $wweb0 $wwel0 -side top

  set wwel [label $wwe.la -text " Really to exit? " -bg "#ffa040" -height 2]
  set wwebs [frame $wwe.bs]
  set wweb1 [button $wwebs.b1 -text "Yes" -width 3 -padx 8 -bd 1 -command exit]
  set wweb2 [button $wwebs.b2 -text "No" -width 3 -padx 8 -bd 1 -command exReturn]
  pack $wweb1 $wweb2 -side left
}
prepareFile $ww


set isAsking 0
proc wantToExit {} {

  DrawFile
  exQuest

}

proc exQuest {} {
  global wweb0 wwel0
  global wwel wwebs
  global isAsking

  set isAsking 1

  pack forget $wwel0
  pack forget $wweb0
  pack $wwel $wwebs -side top

}

proc exReturn {} {
  global wweb0 wwel0
  global wwel wwebs
  global isAsking

  set isAsking 0

  pack forget $wwebs
  pack forget $wwel
  pack $wweb0 $wwel0 -side top

}

proc prLoad {} {
  global wwi1l1

  set txtOK "load: successful"
  set txtKO "load: failed"

  set rv [prLoadInner]
  if {0 > $rv} {
    $wwi1l1 configure -text " $txtKO "
    return 0
  } else {
    $wwi1l1 configure -text " $txtOK "
    return 1
  }

}

proc prLoadInner {} {
  global prDefFile

  global probsCounts
  global probsSingle
  global probsBonfer

  set probsCounts [list]
  set probsSingle [list]
  set probsBonfer [list]

  updateEstims

  set tmpCounts [list]
  set tmpSingle [list]
  set tmpBonfer [list]

  set isAvail 1
  if {0 == [string length $prDefFile]} {set isAvail 0}
  if {0 == [file exists $prDefFile]} {set isAvail 0}
  if {0 == [file isfile $prDefFile]} {set isAvail 0}
  if {0 == $isAvail} {
    return -1
  }

  set ierr [catch {set fl [open $prDefFile "r"]}]
  if {0 != $ierr} {
    return -2
  }
  set line ""

  set state "start"
  while {[gets $fl line] >= 0} {
    set cline [string trim $line]
    if {0 == [string length $cline]} {
      continue
    }
    if {1 == [string equal -length 1 "#" $cline]} {
      continue
    }

    set clist [list]
    set clist0 [split $cline ","]
    foreach elm $clist0 {
      lappend clist [string trim $elm]
    }

    set clen [llength $clist]
    if {7 > $clen} {
      close $fl
      return -3
    }

    if {"start" == $state} {
      set cnames [list "probs" "e.1" "s.1" "d.1" "e.b" "s.b" "d.b"]
      for {set ind 0} {$ind < 7} {incr ind} {
        set req [lindex $cnames $ind]
        set val [lindex $clist $ind]
        if {$req != $val} {
          close $fl
          return -4
        }
      }

      set state "probs"
      continue
    }

    set cline0 [lindex $clist 0]
    if {0 == [string is digit -strict $cline0]} {
      close $fl
      return -5
    }

    for {set ind 1} {$ind < 7} {incr ind} {
      set cval [lindex $clist $ind]
      set cval$ind $cval
      if {0 == [string is double -strict $cval]} {
        close $fl
        return -6
      }
    }

    lappend tmpCounts $cline0
    lappend tmpSingle [list $cval1 $cval2 $cval3]
    lappend tmpBonfer [list $cval4 $cval5 $cval6]

  }

  close $fl

  if {0 == [llength $tmpCounts]} {
    return -7
  }

  set probsCounts $tmpCounts
  set probsSingle $tmpSingle
  set probsBonfer $tmpBonfer

  updateEstims

  return 1

}


proc frLoad {} {
  global wwi2l1

  set txtOK "load: successful"
  set txtKO "load: failed"

  set rv [frLoadInner]
  if {0 > $rv} {
    $wwi2l1 configure -text " $txtKO "
    return 0
  } else {
    $wwi2l1 configure -text " $txtOK "
    return 1
  }

}

proc frLoadInner {} {
  global frDefFile
  global inccaselen
  global inccases
  global inctuplets
  global incgapslen
  global incgaplist
  global incrowlen
  global incrowvis
  global incinilist
  global inccurlist

  global incusegap

  global sortedesd
  global sorteddse
  global sortedsde
  global sortedsed

  global tuploff
  global tuploffscale
  global sorttype
  global selgap
  global selgapbox
  global gaplabel

  set inccaselen 0
  set inccases [list]
  set inctuplets [list]
  set incgapslen 0
  set incgaplist [list]
  set incrowlen 0
  set incinilist [list]
  set inccurlist [list]

  set incusegap -1

  array unset sortedesd
  array unset sorteddse
  array unset sortedsde
  array unset sortedsed

  set tuploff 0
  $tuploffscale configure -to 0
  set sorttype 0
  set selgap [list]
  $gaplabel configure -text "--"
  setTuplOffset -1

  set isAvail 1
  if {0 == [string length $frDefFile]} {set isAvail 0}
  if {0 == [file exists $frDefFile]} {set isAvail 0}
  if {0 == [file isfile $frDefFile]} {set isAvail 0}
  if {0 == $isAvail} {
    return -1
  }

  set ierr [catch {set fl [open $frDefFile "r"]}]
  if {0 != $ierr} {
    return -2
  }
  set line ""

  set tmpcaselen 0
  set tmpcases [list]
  set tmptuplets [list]
  set tmpgapslen 0
  set tmpgaplist [list]

  set state "start"
  while {[gets $fl line] >= 0} {
    set cline [string trim $line]
    if {0 == [string length $cline]} {
      continue
    }
    if {1 == [string equal -length 1 "#" $cline]} {
      continue
    }

    set clist [list]
    set clist0 [split $cline ","]
    foreach elm $clist0 {
      lappend clist [string trim $elm]
    }

    set clen [llength $clist]
    if {"start" == $state} {
      if {2 > $clen} {
        close $fl
        return -3
      }
      set cline0 [string tolower [lindex $clist 0]]
      if {"cases" == $cline0} {
        set tmpcaselen $clen
        if {0 == [expr {($clen - 1) % 3}]} {
          set tmpgapslen [expr {($clen - 1) / 3}]
        }
      }
      set okcur 0
      if {0 < $tmpgapslen} {
        set okcur 1
        foreach {ve vs vd} [lrange $clist 1 end] {
          if {0 == $okcur} {
            continue
          }
          set auxgap -1
          set velist [split $ve "."]
          set vslist [split $vs "."]
          set vdlist [split $vd "."]
          if {2 != [llength $velist]} {
            set okcur 0
          }
          if {2 != [llength $vslist]} {
            set okcur 0
          }
          if {2 != [llength $vdlist]} {
            set okcur 0
          }
          if {1 == $okcur} {
            if {"e" != [lindex $velist 0]} {
              set okcur 0
            }
            set auxgap [lindex $velist 1]
            if {0 == [string is digit -strict $auxgap]} {
              set okcur 0
            }
          }
          if {1 == $okcur} {
            if {"s" != [lindex $vslist 0]} {
              set okcur 0
            }
            if {"d" != [lindex $vdlist 0]} {
              set okcur 0
            }
            if {$auxgap != [lindex $vslist 1]} {
              set okcur 0
            }
            if {$auxgap != [lindex $vdlist 1]} {
              set okcur 0
            }
          }
          if {1 == $okcur} {
            lappend tmpgaplist $auxgap
          }
        }
      }
      if {1 == $okcur} {
        set state "cases"
      }

      if {"cases" != $state} {
        close $fl
        return -4
      }

      continue
    }

    if {$tmpcaselen != $clen} {
      close $fl
      return -5
    }

    set auxlist [lrange $clist 1 end]
    foreach val $auxlist {
      if {0 == [string is digit -strict $val]} {
        close $fl
        return -6
      }
    }
    lappend tmpcases $auxlist
    lappend tmptuplets [lindex $clist 0]

  }

  close $fl

  set tmprowlen [llength $tmptuplets]
  if {0 == $tmprowlen} {
    return -7
  }

  for {set ind 0} {$ind < $tmpgapslen} {incr ind} {
    set auxlist1 [list]
    set auxlist2 [list]
    set auxlist3 [list]
    set auxlist4 [list]
    set ive [expr {3 * $ind}]
    set ivs [expr {1 + $ive}]
    set ivd [expr {1 + $ivs}]
    set curind 0
    foreach onecase $tmpcases {
      set ve [lindex $onecase $ive]
      set vs [lindex $onecase $ivs]
      set vd [lindex $onecase $ivd]
      lappend auxlist1 [list $curind "$ve:$vs:$vd"]
      lappend auxlist2 [list $curind "$vd:$vs:$ve"]
      lappend auxlist3 [list $curind "$vs:$vd:$ve"]
      lappend auxlist4 [list $curind "$vs:$ve:$vd"]
      incr curind
    }

    set auxsorted1 [lsort -dictionary -index 1 -decreasing $auxlist1]
    set auxsorted2 [lsort -dictionary -index 1 -decreasing $auxlist2]
    set auxsorted3 [lsort -dictionary -index 1 -decreasing $auxlist3]
    set auxsorted4 [lsort -dictionary -index 1 -decreasing $auxlist4]

    set tmpaux [list]
    foreach curitem $auxsorted1 {
      set sind [lindex $curitem 0]
      lappend tmpaux $sind
    }
    array set sortedesd [list $ind $tmpaux]

    set tmpaux [list]
    foreach curitem $auxsorted2 {
      set sind [lindex $curitem 0]
      lappend tmpaux $sind
    }
    array set sorteddse [list $ind $tmpaux]

    set tmpaux [list]
    foreach curitem $auxsorted3 {
      set sind [lindex $curitem 0]
      lappend tmpaux $sind
    }
    array set sortedsde [list $ind $tmpaux]

    set tmpaux [list]
    foreach curitem $auxsorted4 {
      set sind [lindex $curitem 0]
      lappend tmpaux $sind
    }
    array set sortedsed [list $ind $tmpaux]

  }

  set inccaselen $tmpcaselen
  set inccases $tmpcases
  set inctuplets $tmptuplets
  set incgapslen $tmpgapslen
  set incgaplist $tmpgaplist
  set incrowlen $tmprowlen

  set incusegap 0

  set incinilist [list]
  for {set ind 0} {$ind < $incrowlen} {incr ind} {
    lappend incinilist $ind
  }
  set inccurlist $incinilist

  set offtoval 0
  if {$incrowlen > $incrowvis} {
    set offtoval [expr {$incrowlen - $incrowvis}]
  }
  $tuploffscale configure -to $offtoval

  set selgap $incgaplist
  $selgapbox selection clear 0 end
  $selgapbox see 0
  $selgapbox activate 0
  $selgapbox selection set 0

  set gval [lindex $incgaplist 0]
  $gaplabel configure -text "$gval"
  setTuplOffset 0

  return 1
}


proc sqLoad {} {
  global wwi3l1

  set txtOK "load: successful"
  set txtKO "load: failed"

  set rv [sqLoadInner]
  if {0 > $rv} {
    $wwi3l1 configure -text " $txtKO "
    return 0
  } else {
    $wwi3l1 configure -text " $txtOK "
    return 1
  }

}


set seqheads [list]
set seqlines [list]
set seqstarts [list]
proc sqLoadInner {} {
  global sqDefFile
  global nwbtxt
  global nwbpos
  global nwb
  global seqheads
  global seqlines
  global seqstarts

  global sqseen
  global sqlate

  global totLines
  global gotpx
  global bfind

  global totFound
  global posFound
  global rnkFound

  global tupStartsCon
  global tupStartsSep

  set totFound 0
  set posFound [list]
  set rnkFound [list]
  updateBSeek

  set tupStartsCon [list]
  set tupStartsSep [list]

  set gotpx 0
  $bfind configure -state "disabled"

  $nwbtxt configure -state "normal"

  deleteLookups
  $nwbtxt delete 1.0 end
  makeGauge 0
  $nwbtxt configure -state "disabled"

  set totLines 0

  set seqlines [list]
  set seqheads [list]
  set seqstarts [list]

  set tmplines [list]
  set tmpheads [list]

  set maxlen 0

  set isAvail 1
  if {0 == [string length $sqDefFile]} {set isAvail 0}
  if {0 == [file exists $sqDefFile]} {set isAvail 0}
  if {0 == [file isfile $sqDefFile]} {set isAvail 0}
  if {0 == $isAvail} {
    return -1
  }

  set ierr [catch {set fl [open $sqDefFile "r"]}]
  if {0 != $ierr} {
    return -2
  }

  set line ""

  set cstate "outside"
  set curseq [list]
  set curind [list]

  set curlen 0
  set globeven 0
  while {[gets $fl line] >= 0} {
    set line [string trim $line]
    if {0 == [string length $line]} {
      continue
    }
    if {0 == [string compare -length 1 "#" $line]} {
      continue
    }
    if {0 == [string compare -length 1 ">" $line]} {
      lappend tmpheads [string range $line 0 499]
    }
    if {0 != [string compare -length 1 ">" $line]} {
      if {0 != [string compare -length 1 "#" $line]} {
        if {"outside" == $cstate} {
          close $fl
          return -3
        }
        if {0 == [string is alnum $line]} {
          close $fl
          return -4
        }

        lappend curseq $line
        incr curlen [string length $line]

      }
    } else {
      if {"outside" != $cstate} {
        lappend tmplines $curseq
        set curseq [list]

        if {$maxlen < $curlen} {
          set maxlen $curlen
        }
        set curlen 0

      } else {
        set cstate "inside"
      }
    }

  }

  if {"outside" != $cstate} {
    lappend tmplines $curseq

    if {$maxlen < $curlen} {
      set maxlen $curlen
    }
  }

  close $fl

  if {0 == [llength $tmpheads]} {
    return -5
  }

  set seqlines $tmplines
  set seqheads $tmpheads
  foreach onehead $seqheads {
    set onestart ""
    set curhead [string range $onehead 1 end]
    set onelist [string trim [split $curhead "|"]]
    set startind 0
    foreach onepart $onelist {
      incr startind
      set descstr [string trim $onepart]
      if {0 == [string compare -nocase $descstr "start"]} {
        set auxstr [lindex $onelist $startind]
        set auxval [string trim $auxstr]
        if {0 == [string length $auxval]} {
          continue
        }
        if {18 < [string length $auxval]} {
          continue
        }

        if {1 == [string is ascii -strict $auxval]} {
          set startval $auxval
          set startdir "+"
          if {0 == [string compare -length 1 "-" $auxval]} {
            set startval [string range $auxval 1 end]
            set startdir "-"
          }
          if {1 == [string is digit -strict $startval]} {
            set startval [string trimleft $startval "0"]
            if {"" == $startval} {
              continue
            }
            if {"-" == $startdir} {
              set startval "-$startval"
            }
            set onestart $startval
            break
          }
        }

      }

    }

    lappend seqstarts $onestart
  }

  $nwbtxt xview moveto 0.0
  $nwbpos xview moveto 0.0
  set curwlist [$nwbtxt xview]
  $nwb.scx set [lindex $curwlist 0] [lindex $curwlist 1]

  dispSeqs

  if {0 != $sqseen} {
    makeGauge [expr {($maxlen / 10) + 1}]
  } else {
    makeGauge 0
    set sqlate [expr {($maxlen / 10) + 1}]
  }

  updateBFind
  return 1

}


proc DrawFile {} {
  global topf
  global pfcurrent
  global ww
  global isAsking
  global wwi1l1 wwi2l1 wwi3l1
  global fmenu
  global btlist

  global topname
  global topw

  $wwi1l1 configure -text " "
  $wwi2l1 configure -text " "
  $wwi3l1 configure -text " "

  if {1 == $isAsking} {
    return
  }

  wm title $topw $topname

  foreach butt $btlist {
    $fmenu.$butt configure -relief "raised"
  }
  $fmenu.mfile configure -relief "sunken"

  if {"none" != $pfcurrent} {
    pack forget $pfcurrent
  }
  set pfcurrent $ww
  pack $ww -expand yes -fill none -side top


  return

}


#-----------------------------------------

DrawAbout


